<?php

namespace Incevio\Package\Wallet\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\PaymentMethod;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\RedirectResponse;
use App\Contracts\PaymentServiceContract;
use App\Services\Payments\PaymentService;
use Incevio\Package\Wallet\Models\Transaction;
use App\Services\Payments\PaypalPaymentService;
use Incevio\Package\Wallet\Notifications\Deposit;
use Incevio\Package\Wallet\Traits\HasTransaction;
use Incevio\Package\Wallet\Jobs\SendNotificationJob;
use Incevio\Package\Wallet\Http\Requests\DepositRequest;

class DepositController extends Controller
{
    use HasTransaction;

    private $wallet;

    /**
     * @param Request $request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function show_form(Request $request)
    {
        $paymentMethods = PaymentMethod::find(get_from_option_table('wallet_payment_methods', []));

        if (Auth::guard('customer')->check()) {
            $tab = 'wallet';
            $customer = Auth::guard('customer')->user();

            // View loaded from theme directory, need to do in better ways
            $content = view('wallet::customer.deposit', compact('paymentMethods', 'customer'))->render();

            return view('theme::dashboard', compact('tab', 'content'));
        }

        if (Auth::user()->isMerchant()) {
            $merchant = Auth::user()->shop;

            return view('wallet::deposit', compact('paymentMethods', 'merchant'));
        }

        return redirect()->back()->with('error', trans('wallet::lang.owner_invalid'));
    }

    /**
     * @param DepositRequest $request
     * @param PaymentServiceContract $paymentService
     * @return RedirectResponse
     */
    public function deposit(DepositRequest $request, PaymentServiceContract $paymentService)
    {
        try {
            $result = $paymentService->setConfig()
                ->setAmount($request->amount)
                ->setDescription(trans('wallet::lang.deposit_description', [
                    'marketplace' => get_platform_title(),
                    'payment_method' => $request->payment_method
                ]))
                ->setConfig()
                ->charge();
        } catch (\Exception $e) {
            Log::error('Payment failed:: ');
            Log::info($e);

            return redirect()->route(self::getRouteName())
                ->with('error', $e->getMessage())->withInput();
        }

        // Check if the result is a RedirectResponse of Paypal and some other gateways
        if ($result instanceof RedirectResponse) {
            return $result;
        }

        // Payment succeed
        if ($result->status == PaymentService::STATUS_PAID) {
            return $this->depositCompleted($request->amount, $this->getMetaInfo($request->payment_method));
        }

        return redirect()->route(self::getRouteName())
            ->with('error', trans('wallet::lang.payment_failed'))->withInput();
    }

    /**
     * @param Request $request
     * @param PaypalExpressPaymentService $paymentService
     * @return RedirectResponse
     */
    public function paypalPaymentSuccess(Request $request, PaypalPaymentService $paymentService)
    {
        if (!$request->has('token') || !$request->has('paymentId') || !$request->has('PayerID')) {
            return redirect()->route('wallet.deposit.failed');
        }

        try {
            $paymentService = $paymentService->paymentExecution($request->get('paymentId'), $request->get('PayerID'));
            //dd($paymentService);
        } catch (\Exception $e) {
            Log::error('Payment failed:: ');
            Log::info($e->getMessage());

            return redirect()->route(self::getRouteName())
                ->with('error', $e->getMessage())->withInput();
        }

        // Payment succeed
        if ($paymentService->status == PaymentService::STATUS_PAID && $paymentService->response) {
            $amount = $paymentService->response->transactions[0]->amount->total;

            return $this->depositCompleted($amount, $this->getMetaInfo("Paypal"));
        }

        return redirect()->route(self::getRouteName())
            ->with('error', trans('wallet::lang.payment_failed'))->withInput();
    }

    /**
     * @param Request $request
     * @return RedirectResponse
     */
    public function paymentFailed(Request $request)
    {
        return redirect()->route(self::getRouteName())
            ->with('error', trans('wallet::lang.payment_failed'))->withInput();
    }

    /**
     * Complete the deposit fund
     *
     * @param $amount
     * @param array $meta
     * @param bool $confirm
     * @return RedirectResponse
     */
    private function depositCompleted($amount, $meta = [], $confirm = true)
    {
        try {
            $meta = array_merge([
                'type' => Transaction::TYPE_DEPOSIT,
            ], $meta);

            $wallet = $this->wallet ?? self::getWallet();

            $trans = $wallet->deposit($amount, $meta, $confirm);
        } catch (\Exception $e) {
            return redirect()->route(self::getRouteName())
                ->with('error', trans('wallet::lang.payment_failed'))->withInput();
        }

        SendNotificationJob::dispatch($trans, Deposit::class);

        return redirect()->route(self::getRouteName('wallet'))
            ->with('success', trans('wallet::lang.payment_success'));
    }

    /**
     * Paystack success:
     * @param Request $request
     * @return RedirectResponse
     */
    public function paystackPaymentSuccess(Request $request)
    {
        if ($request->has('trxref') && $request->has('reference')) {
            $paymentService = new \Incevio\Package\Paystack\Services\PaystackPaymentService($request);

            $response = $paymentService->setConfig()->verifyPaidPayment();

            // If the payment failed
            if ($response->status == PaymentService::STATUS_PAID) {
                return $this->depositCompleted($response->amount, $this->getMetaInfo("Paystack"));
            }
        }

        return redirect()->route(self::getRouteName())
            ->with('error', trans('wallet::lang.payment_failed'))->withInput();
    }

    /**
     * Sslcommerz success:
     * @param Request $request
     * @return RedirectResponse
     */
    public function sslcommerzPaymentSuccess(Request $request)
    {
        $paymentService = new \Incevio\Package\SslCommerz\Services\SslCommerzPaymentService($request);

        if ($paymentService->verifyPaidPayment()) {
            return $this->depositCompleted($request->amount, $this->getMetaInfo("sslcommerz"));
        }

        return redirect()->route(self::getRouteName())
            ->with('error', trans('wallet::lang.payment_failed'))->withInput();
    }

    /**
     * flutterwave success:
     * @param Request $request
     * @return RedirectResponse
     */
    public function flutterwavePaymentRedirect(Request $request)
    {
        if ($request->status == 'successful') {
            $flutter = new \Incevio\Package\FlutterWave\Services\FlutterWavePaymentService($request);
            $response = $flutter->verifyPaidPayment();

            if ($response->status == PaymentService::STATUS_PAID) {
                return $this->depositCompleted($response->amount, $this->getMetaInfo("flutterwave"));
            }
        }

        return redirect()->route(self::getRouteName())
            ->with('error', trans('wallet::lang.payment_failed'))->withInput();
    }

    /**
     * mollie success:
     * @param Request $request
     * @return RedirectResponse
     */
    public function molliePaymentRedirect(Request $request)
    {
        $mollie = new \Incevio\Package\Mollie\Services\MolliePaymentService($request);
        $mollie->setConfig();
        $response = $mollie->verifyPaidPayment();

        if ($response->status == PaymentService::STATUS_PAID) {
            return $this->depositCompleted($response->amount, $this->getMetaInfo("mollie"));
        }

        return redirect()->route(self::getRouteName())
            ->with('error', trans('wallet::lang.payment_failed'))->withInput();
    }

    /**
     * bkash success:
     * @param Request $request
     * @return RedirectResponse
     */
    public function bkashPaymentRedirect(Request $request)
    {
        $bkash = new \Incevio\Package\Bkash\Services\BkashPaymentService($request);
        $bkash->setConfig();
        $response = $bkash->verifyPaidPayment();

        if ($response->status == PaymentService::STATUS_PAID) {
            return $this->depositCompleted($response->amount, $this->getMetaInfo("bkash"));
        }

        return redirect()->route(self::getRouteName())
            ->with('error', trans('wallet::lang.payment_failed'))->withInput();
    }

    /**
     * paytm success:
     * @param Request $request
     * @return RedirectResponse
     */
    public function paytmPaymentRedirect(Request $request)
    {
        $paytm = new \Incevio\Package\Paytm\Services\PaytmPaymentService($request);
        $paytm->setConfig();
        $response = $paytm->verifyPaidPayment();

        if ($response->status == PaymentService::STATUS_PAID) {
            return $this->depositCompleted($response->amount, $this->getMetaInfo('paytm'));
        }

        return redirect()->route(self::getRouteName())
            ->with('error', trans('wallet::lang.payment_failed'))->withInput();
    }

    /**
     * Receive payfast payment notification callback
     * 
     * @param Request $request
     * @return RedirectResponse
     */
    public function payfastPaymentNotify(Request $request)
    {
        $pfData = $request->all();

        $payfast = new \Incevio\Package\Payfast\Services\PayfastPaymentService($request);

        $response = $payfast->verifyPaidPayment($pfData);

        if (
            $response->status == PaymentService::STATUS_PAID &&
            $pfData['custom_str2'] && $pfData['email_address']
        ) {
            // Set the wallet for the deposit
            $this->wallet = self::getWallet($pfData['custom_str2'], $pfData['email_address']);

            $this->depositCompleted($response->amount, $this->getMetaInfo("payfast"));
        }

        return response('success!', 200);
    }

    /**
     * Instamojo success:
     * @param Request $request
     * @return RedirectResponse
     */
    public function instamojoPaymentSuccess(Request $request)
    {
        $instamojo = new \Incevio\Package\Instamojo\Services\InstamojoPaymentService($request);
        $response = $instamojo->setConfig()->verifyPaidPayment();

        if ($response->status == PaymentService::STATUS_PAID) {
            return $this->depositCompleted($response->amount, $this->getMetaInfo("instamojo"));
        }

        return redirect()->route(self::getRouteName())
            ->with('error', trans('wallet::lang.payment_failed'))->withInput();
    }

    /**
     * return formated meta info for the transection
     *
     * @return str
     */
    private function getMetaInfo($payment)
    {
        return [
            'description' => trans('wallet::lang.deposit_description', [
                'marketplace' => get_platform_title(),
                'payment_method' => $payment
            ])
        ];
    }
}
